import {useState, useEffect, useRef, createRef} from "react";
import DatePicker, { DateObject, DatePickerProps as RMDPProps } from "react-multi-date-picker";  // Import DateObject
import { FormInput } from "@/components/Base/Form";

// Extend the DatePickerProps to include additional properties
interface CustomDatePickerProps extends RMDPProps {
    format?: string;
    calendar?: string;  // Calendar as a string type (e.g., 'persian', 'arabic')
    locale?: string;  // Locale as a string type (e.g., 'persian_fa', 'arabic_ar')
    numberOfMonths?: number;
    range?: boolean;
    dateSeparator?: string;
}

export interface DatePickerProps
    extends React.PropsWithChildren,
        Omit<React.ComponentPropsWithoutRef<"input">, "onChange"> {
    options?: CustomDatePickerProps;  // Use the props from the package
    onChange: (e: {
        target: {
            value: string;
        };
    }) => void;
    value?: string;
    getRef?: (el: HTMLInputElement) => void;
}

// Function to dynamically load calendar and locale modules based on strings
async function loadCalendarAndLocale(calendar: string, locale: string) {
    let calendarModule;
    let localeModule;

    // Dynamically load calendar based on string input
    switch (calendar) {
        case "gregorian":
            calendarModule = (await import("react-date-object/calendars/gregorian")).default;
            break;
        case "arabic":
            calendarModule = (await import("react-date-object/calendars/arabic")).default;
            break;
        default:
            calendarModule = (await import("react-date-object/calendars/persian")).default;
    }

    // Dynamically load locale based on string input
    switch (locale) {
        case "gregorian_en":
            localeModule = (await import("react-date-object/locales/gregorian_en")).default;
            break;
        case "arabic_ar":
            localeModule = (await import("react-date-object/locales/arabic_ar")).default;
            break;
        default:
            localeModule = (await import("react-date-object/locales/persian_fa")).default;
    }

    return { calendarModule, localeModule };
}

// Define the correct types for the calendar and locale
interface CalendarAndLocale {
    calendar: any;
    locale: any; // Since there's no explicit 'Locale' type, use 'any' for locale
}

function MultiDatePicker({
                              options = {},
                              value = "",
                              onChange = () => {},
                              getRef = () => {},
                              ...computedProps
                          }: DatePickerProps) {
    const [selectedDate, setSelectedDate] = useState<DateObject | null>(
        value ? new DateObject(value) : null  // Create a new DateObject instance
    );

    const props = {
        options: options,
        value: value,
        onChange: onChange,
        getRef: getRef,
    };

    const initialRender = useRef(true);
    const rmdpRef = createRef<HTMLInputElement>();
    const tempValue = useRef(value);

    const [calendarAndLocale, setCalendarAndLocale] = useState<CalendarAndLocale>({
        calendar: undefined,
        locale: undefined,
    });

    useEffect(() => {
        // Dynamically load the correct calendar and locale based on the options
        async function fetchCalendarAndLocale() {
            const { calendarModule, localeModule } = await loadCalendarAndLocale(
                options.calendar || "persian", // Default to gregorian if no calendar is provided
                options.locale || "fa" // Default to English locale
            );

            setCalendarAndLocale({
                calendar: calendarModule,
                locale: localeModule,
            });
        }

        fetchCalendarAndLocale();
    }, [options.calendar, options.locale]);

    useEffect(() => {
        // Synchronize internal state with incoming `value` prop
        if (calendarAndLocale.calendar &&
            calendarAndLocale.locale &&
            value && selectedDate?.format(options.format || "YYYY-MM-DD") !== value) {
            setSelectedDate(new DateObject({
                date: value,
                calendar: calendarAndLocale.calendar,
                locale: calendarAndLocale.locale,
            }));  // Update the selectedDate state
        }
    }, [value, options.format, selectedDate]);

    useEffect(() => {
        if (rmdpRef.current) {
            props.getRef(rmdpRef.current);
        }

        if (initialRender.current) {
            setSelectedDate(new DateObject({
                date: value,
                calendar: calendarAndLocale.calendar,
                locale: calendarAndLocale.locale,
            }));
            initialRender.current = false;
        }

        tempValue.current = props.value;
    }, [props.value]);

    if (!calendarAndLocale.calendar || !calendarAndLocale.locale) {
        return <div>بارگذاری...</div>; // Show a loading state until the calendar and locale are ready
    }

    return (
        <DatePicker
            value={selectedDate}  // Pass the selectedDate object to the DatePicker
            onChange={(date: DateObject) => {  // Handle onChange event from DatePicker
                setSelectedDate(date);  // Update state when a new date is selected
                onChange({
                    target: {
                        value: date?.format(options.format || "YYYY-MM-DD"),  // Format the date
                    },
                });
            }}
            format={options.format || "YYYY-MM-DD"}  // Set the format of the displayed date
            calendar={calendarAndLocale.calendar}  // Dynamically loaded calendar
            locale={calendarAndLocale.locale}  // Dynamically loaded locale
            containerClassName={options?.containerClassName}
            numberOfMonths={options?.numberOfMonths}
            range={options?.range}
            dateSeparator={options?.dateSeparator}
            calendarPosition="bottom-center" // Example positioning
            render={(value, openCalendar) => (
                <FormInput
                    type="text"
                    value={value}
                    onFocus={openCalendar}  // Open the calendar when input is focused
                    {...computedProps}
                    ref={(el) => el && getRef(el)}  // Provide the input element reference
                />
            )}
        />
    );
}

export default MultiDatePicker;
